/*
 *  Copyright (C) 2002, 2003 Tomasz Kojm <zolw@konarski.edu.pl>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/time.h>
#include <time.h>

#include "options.h"
#include "others.h"
#include "shared.h"
#include "manager.h"
#include "defaults.h"
#include "treewalk.h"
#include "clamav.h"

void help(void);

/* this local macro takes care about freeing memory at exit */
/*
#define mexit(i)    if(opt) free_opt(opt);			    \
		    mprintf("*Memory freed. Exit code: %d\n", i);   \
		    exit(i);
*/
#define mexit(i)    exit(i)

int isnumb(const char *str)
{
	int i;

    for(i = 0; i < strlen(str); i++)
	if(!isdigit(str[i]))
	    return 0;

    return 1;
}

void clamscan(struct optstruct *opt)
{
	int ds, dms, ret;
	double mb;
	struct timeval t1, t2;
	struct timezone tz;
	time_t starttime;


    /* initialize some important variables */

    mprintf_disabled = 0;

    if(optc(opt, 'v')) mprintf_verbose = 1;
    else mprintf_verbose = 0;

    if(optl(opt, "quiet")) mprintf_quiet = 1;
    else mprintf_quiet = 0;

    if(optl(opt, "stdout")) mprintf_stdout = 1;
    else mprintf_stdout = 0;

    if(optc(opt, 'V')) {
	printf("clamscan / ClamAV version "VERSION"\n");
	mexit(0);
    }

    if(optc(opt, 'h')) {
	free_opt(opt);
    	help();
    }

    /* check other options */

    if(optc(opt, 'r')) recursion = 1;
    else recursion = 0;

    if(optc(opt, 'i')) printinfected = 1;
    else printinfected = 0;

    /* initialize logger */

    if(optl(opt, "log-verbose")) logverbose = 1;
    else logverbose = 0;

	/*
    if(optc(opt, 'l')) {
	logfile = getargc(opt, 'l');
	if(logg("--------------------------------------\n")) {
	    printf("!Problem with internal logger.\n");
	    mexit(1);
	}
    } else 
	logfile = NULL;
	*/

    /* we need some pre-checks */
    if(optl(opt, "max-space"))
	if(!strchr(getargl(opt, "max-space"), 'M') && !strchr(getargl(opt, "max-space"), 'm'))
	    if(!isnumb(getargl(opt, "max-space"))) {
		printf("!--max-space requires natural number.\n");
		exit(40);
	    }

    if(optl(opt, "max-files"))
	if(!isnumb(getargl(opt, "max-files"))) {
	    printf("!--max-files requires natural number.\n");
	    exit(40);
	}

    if(optl(opt, "max-recursion"))
	if(!isnumb(getargl(opt, "max-recursion"))) {
	    printf("!--max-recursion requires natural number.\n");
	    exit(40);
	}


	time(&starttime);
	/* ctime() does \n, but I need it once more */
	//logg("Scan started: %s\n", ctime(&starttime));

	memset(&claminfo, 0, sizeof(struct s_info));

	gettimeofday(&t1, &tz);
	ret = scanmanager(opt);

	if(!optl(opt, "disable-summary")) {
		gettimeofday(&t2, &tz);
		ds = t2.tv_sec - t1.tv_sec;
		dms = t2.tv_usec - t1.tv_usec;
		ds -= (dms < 0) ? (1):(0);
		dms += (dms < 0) ? (1000000):(0);
		printf("\n----------- SCAN SUMMARY -----------\n");
		//logg("\n-- summary --\n");
		printf("Known viruses: %d\n", claminfo.signs);
		//logg("Known viruses: %d\n", claminfo.signs);
		printf("Scanned directories: %d\n", claminfo.dirs);
		//logg("Scanned directories: %d\n", claminfo.dirs);
		printf("Scanned files: %d\n", claminfo.files);
		//logg("Scanned files: %d\n", claminfo.files);
		printf("Infected files: %d\n", claminfo.ifiles);
		//logg("Infected files: %d\n", claminfo.ifiles);
		if(claminfo.notremoved) {
			printf("Not removed: %d\n", claminfo.notremoved);
			//logg("Not removed: %d\n", claminfo.notremoved);
		}
		if(claminfo.notmoved) {
			printf("Not moved: %d\n", claminfo.notmoved);
			//logg("Not moved: %d\n", claminfo.notmoved);
		}
		mb = claminfo.blocks * (CL_COUNT_PRECISION / 1024) / 1024.0;
		printf("Data scanned: %2.2lf Mb\n", mb);
		//logg("Data scanned: %2.2lf Mb\n", mb);

		printf("I/O buffer size: %d bytes\n", BUFFSIZE);
		//logg("I/O buffer size: %d bytes\n", BUFFSIZE);
		printf("Time: %d.%3.3d sec (%d m %d s)\n", ds, dms/1000, ds/60, ds%60);
		//logg("Time: %d.%3.3d sec (%d m %d s)\n", ds, dms/1000, ds/60, ds%60);
	}

    mexit(ret);
}

void help(void)
{

    mprintf_stdout = 1;

    printf("\n");
    printf("		   Clam AntiVirus Scanner "VERSION"\n");
    printf("		   (c) 2002 Tomasz Kojm <zolw@konarski.edu.pl>\n");
    printf("	  \n");
    printf("    --help		    -h		Show help\n");
    printf("    --version		    -V		Print version number and exit\n");
    printf("    --verbose		    -v		Be verbose\n");
    printf("    --quiet				Be quiet, output only error messages\n");
    printf("    --stdout				Write to stdout instead of stderr\n");
    printf("					(this help is always written to stdout)\n");
    printf("    --force				Try to ignore some errors\n");
    printf("\n");
    printf("    --tempdir=DIRECTORY			create temporary files in DIRECTORY\n");
    printf("    --database=FILE/DIR    -d FILE/DIR	Load virus database from FILE or load\n");
    printf("					all .db and .db2 files from DIR\n");
    printf("    --log=FILE		    -l FILE	Save scan report in FILE\n");
    printf("    --log-verbose			Save additional informations\n");
    printf("    --recursive    	    -r	    	Scan directories recursively\n");
    printf("    --infected		    -i		Print infected files only\n");
    printf("    --remove				Remove infected files. Be careful.\n");
    printf("    --move=DIRECTORY			Move infected files into DIRECTORY\n");
    printf("    --exclude=PATT			Don't scan file names containing PATT\n");
    printf("    --include=PATT			Only scan file names containing PATT\n");
    printf("    --disable-summary			Disable summary at end of scanning\n");
    printf("    --mbox		    -m		Treat stdin as a mailbox\n");
    printf("\n");
    printf("    --disable-archive			Disable libclamav archive support\n");
    printf("    --max-space=#n			Extract first #n kilobytes only\n");
    printf("    --max-files=#n			Extract first #n files only\n");
    printf("    --max-recursion=#n			Maximal recursion level\n");
    printf("    --unzip[=FULLPATH]			Enable support for .zip files\n");
    printf("    --unrar[=FULLPATH]			Enable support for .rar files\n");
    printf("    --unace[=FULLPATH]			Enable support for .ace files\n");
    printf("    --arj[=FULLPATH]			Enable support for .arj files\n");
    printf("    --unzoo[=FULLPATH]			Enable support for .zoo files\n");
    printf("    --lha[=FULLPATH]			Enable support for .lha files\n");
    printf("    --jar[=FULLPATH]			Enable support for .jar files\n");
    printf("    --tar[=FULLPATH]			Enable support for .tar files\n");
    printf("    --deb[=FULLPATH to ar]    		Enable support for .deb files,\n");
    printf("					implies --tgz , but doesn't conflict\n");
    printf("					with --tgz=FULLPATH.\n");
    printf("    --tgz[=FULLPATH]			enable support for .tar.gz, .tgz files\n\n");

    exit(0);
}
